#include <USB.h>
#include <HID.h>
#include <stdlib.h>
#include <string.h>

//#define IOWARRIOR_LIB_DEBUG 1

#ifdef IOWARRIOR_LIB_DEBUG
	#include <stdio.h>
#endif

//#define IOWARRIOR_LIB_ENABLED_NOTIFICATIONS 1

#include "IOWarriorOS9Lib.h"

#define kIOWarriorVendorID 1984
#define kIOWarrior24DeviceID 0x1501
#define kIOWarrior40DeviceID 0x1500


// Global library variables

IOWarriorListNode*	gIOWarriorList; // linked list of IOWarrior interfaces already discovered
short				gIOWarriorListDirty = true; // is the IOWarrior interface list list up-to-date or dirty
int					gIOWarriorCount = 0;
short				gIOWarriorReadOperationPending;
char*				gIOWarriorReadBuffer;
int					gIOWarriorReadBufferSize;
int					gIOWarriorNotificationToken;
short				gIOWarriorAutoCloseInterfaces; // should the used interface be closed after read/write operation, default is yes

// private prototype declarations

void IOWarriorDiscoverInterfaces ();
int IOWarriorCount ();

void 						IOWarriorClearInterfaceList ();

void 						IOWarriorAddInterfaceToList (HIDDeviceDispatchTablePtr inInterface, int inInterfaceType);

HIDDeviceDispatchTablePtr 	IOWarriorInterfaceAtIndex (int inIndex);
IOWarriorListNode* 			IOWarriorListNodeAtIndex (int inIndex);

HIDDeviceDispatchTablePtr IOWarriorFirstInterfaceOfType (int inInterfaceType);

OSErr IOWarriorOpenInterfaceAtListNode (IOWarriorListNode* inNode);
OSErr IOWarriorCloseInterfaceAtListNode  (IOWarriorListNode* inNode);


void IOWarriorHIDReportHandlerProcPtr(void *inHIDReport, UInt32 inHIDReportLength, UInt32 inRefcon);


void IOWarriorNotificationCallback(USBDeviceNotificationParameterBlock *pb);

int IOWarriorInit ()
{
	USBDeviceNotificationParameterBlock pb;
	
	gIOWarriorList = NULL;
    gIOWarriorListDirty = 1;
    gIOWarriorCount = 0;
    gIOWarriorReadOperationPending = false;
    gIOWarriorAutoCloseInterfaces = true;
    
    // Install notification callback so unpluggin/plugging gets noticed
    
    pb.usbDeviceNotification = -1; // tell me about everything
	pb.usbClass = kUSBHIDInterfaceClass; // want to know about HID class devices
	pb.usbSubClass = kUSBAnySubClass; // tell me about all sublclass devices
	pb.usbProtocol = -1; // don't care about the protocol used by device
	pb.usbVendor = kIOWarriorVendorID; // allow only IOWarrior device vendor notification
	pb.usbProduct = kUSBAnyProduct; // allow any product ID notification
	pb.result = noErr;
	pb.callback = (USBDeviceNotificationCallbackProcPtr)&IOWarriorNotificationCallback;
	pb.refcon = nil;
	
	#ifdef IOWARRIOR_LIB_ENABLED_NOTIFICATIONS
	USBInstallDeviceNotification (&pb);
	gIOWarriorNotificationToken = pb.token;
	#endif
    
    /* temporary code until callback functions work */
    #ifndef IOWARRIOR_LIB_ENABLED_NOTIFICATIONS
    IOWarriorDiscoverInterfaces ();
    gIOWarriorListDirty = 0;
    #endif
    
    return 0;
}

void IOWarriorClose ()
{
	USBRemoveDeviceNotification (gIOWarriorNotificationToken);
}

// Returns 1 if at least one IOWarrior is connected to this system. Returns 0 if no IOWarrior device could be discovered.
int IOWarriorIsPresent ()
{
	if (gIOWarriorListDirty)
	{
		IOWarriorClearInterfaceList ();
		IOWarriorDiscoverInterfaces ();
		gIOWarriorListDirty = false;
	}
    return (IOWarriorCount () > 0);
}

int IOWarriorCount ()
{
	return gIOWarriorCount;
}

OSErr IOWarriorOpenInterfaceAtListNode (IOWarriorListNode* inNode)
{
	OSErr err = noErr;

	if (!inNode->isOpen)
	{
		HIDDeviceDispatchTablePtr	interface = inNode->HIDDispatchTable;
		
		err = (*interface->pHIDOpenDevice) (&inNode->connectionRef,kHIDPerm_ReadWriteShared, 0);
		if (noErr == err)
			inNode->isOpen = true;
	}
	#ifdef IOWARRIOR_LIB_DEBUG
				if (err) {	printf ("theHIDDispatchTable->pHIDOpenDevice returned error: %d\n", err );}
	#endif
	return err;
}

OSErr IOWarriorCloseInterfaceAtListNode  (IOWarriorListNode* inNode)
{
	OSErr err = noErr;

	if (inNode->isOpen)
	{
		HIDDeviceDispatchTablePtr	interface = inNode->HIDDispatchTable;
		
		err = (*interface->pHIDCloseDevice) (inNode->connectionRef);
		if (noErr == err)
			inNode->isOpen = false;
	}
	#ifdef IOWARRIOR_LIB_DEBUG
				if (err) {	printf ("theHIDDispatchTable->pHIDCloseDevice returned error: %d\n", err );}
	#endif
	return err;

}


// read inSize bytes from interface inInterfaceIndex of device inWarriorIndex to outData.
int IOWarriorRead (IOWarriorListNode* inNode, int inReportID, int inSize, void* outData)
{
	OSErr						err = noErr;
	HIDDeviceDispatchTablePtr	interface = inNode->HIDDispatchTable;
		
	// open the interface
	err = IOWarriorOpenInterfaceAtListNode (inNode);
	if (noErr == err)
	{
		// write the data
		gIOWarriorReadBuffer = NewPtr (inSize);
		gIOWarriorReadBufferSize = inSize;
		gIOWarriorReadOperationPending = true;
		
		// pHIDGetSizedReport is available from version 3
		if (interface->dispatchTableCurrentVersion >= 3 && interface->pHIDGetSizedReport)
		{
			err = (*interface->pHIDGetSizedReport) ( inNode->connectionRef, kHIDOutputReport, inReportID, inSize, IOWarriorHIDReportHandlerProcPtr , 0);
			if (err)
			{
    			#ifdef IOWARRIOR_LIB_DEBUG
					printf ("theHIDDispatchTable->pHIDGetSizedReport returned error: %d\n", err );
				#endif
			}
		}
		else if (interface->pHIDGetReport)
		{
			err = (*interface->pHIDGetReport) ( inNode->connectionRef, kHIDOutputReport, inReportID, IOWarriorHIDReportHandlerProcPtr , inSize); 
			if (err)
			{
    			#ifdef IOWARRIOR_LIB_DEBUG
					printf ("theHIDDispatchTable->pHIDGetReport returned error: %d\n", err );
				#endif
			}
		}
		if (noErr == err)
		{
			while (true == gIOWarriorReadOperationPending)
			{
				;
			}
			BlockMoveData (gIOWarriorReadBuffer, outData, inSize);
		}
		DisposePtr (gIOWarriorReadBuffer);
		gIOWarriorReadBufferSize = 0;
	}
	if (gIOWarriorAutoCloseInterfaces)
		IOWarriorCloseInterfaceAtListNode (inNode);
	

	return err;
}

// write inSize bytes at inData to interface inInterfaceIndex of device inWarriorIndex.
int IOWarriorWrite (IOWarriorListNode* inNode, int inReportID, int inSize, void* inData)
{
	OSErr						err = noErr;
	HIDDeviceDispatchTablePtr 	interface = inNode->HIDDispatchTable;
	
	
	// open the interface
	err = IOWarriorOpenInterfaceAtListNode (inNode);
	if (noErr == err)
	{
		do
		{
			// try to write the data, repeat whe command queue is full (memFullErr) is returned
			err = (*interface->pHIDSetReport) (inNode->connectionRef, kHIDOutputReport, inReportID, inData, inSize);
			if (err && err != memFullErr)
			{
    			#ifdef IOWARRIOR_LIB_DEBUG
					printf ("theHIDDispatchTable->pHIDSetReport returned error: %d\n", err );
				#endif
			}
		}
		while (memFullErr == err);
	}
	if (gIOWarriorAutoCloseInterfaces)
		IOWarriorCloseInterfaceAtListNode (inNode);
	return err;
}

// writes an 4 byte buffer to interface 0 of the first discovered IOWarrior40
int IOWarriorWriteInterface0 (void *inData)
{
	IOWarriorListNode *theNode;
	
	if (NULL != (theNode = IOWarriorFirstInterfaceNodeOfType (kIOWarrior40Interface0)))
    	return IOWarriorWrite (theNode, 0, 4, inData);
    	
    return -1;
}

// reads 4 byte from interface 0 of the first discovered IOWarrior40
int IOWarriorReadInterface0 (void *outData)
{
	IOWarriorListNode *theNode;
	
	if (NULL != (theNode = IOWarriorFirstInterfaceNodeOfType (kIOWarrior40Interface0)))
    	return IOWarriorRead (theNode, 0, 4, outData);
    	
    return -1;
}

/* Writes an 7 byte buffer to interface 1 of the first discovered IOWarrior40. 
Since passing simply 7 bytes and an reportID fails when doing the setReport call in IOWarriorWrite, 
reportID and data are merged into a single buffer which is passed to IOWarriorWrite using reportID 0 */
int IOWarriorWriteInterface1 (int inReportID, void *inData)
{
	IOWarriorListNode *theNode;
	OSErr				err = noErr;
	
	if (NULL != (theNode = IOWarriorFirstInterfaceNodeOfType (kIOWarrior40Interface1)))
	{
    	char buffer[8];

    	buffer[0] = inReportID;

    	memcpy (&buffer[1], inData, 7);
    	err = IOWarriorWrite (theNode, 0, 8, buffer);
    }
    return err;
}

/* Returns first interface of type inType. */
HIDDeviceDispatchTablePtr IOWarriorFirstInterfaceOfType (int inInterfaceType)
{
	IOWarriorListNode* node;
	
	node = IOWarriorListNodeAtIndex (inInterfaceType);
	if (NULL != (node = IOWarriorFirstInterfaceNodeOfType (inInterfaceType)))
	{
		return node->HIDDispatchTable;
	}
	return NULL;
}

IOWarriorListNode* IOWarriorFirstInterfaceNodeOfType (int inInterfaceType)
{
	int i, limit;
	
	limit = IOWarriorCount ();
	for (i = 0; i < limit; i++)
	{
		IOWarriorListNode* node;
		
		node = IOWarriorListNodeAtIndex (i);
		if (node && node->interfaceType == inInterfaceType)
		{
			return node;
		}
	}
	return NULL;
}


/* Populates the linked list of known IOWarrior Interfaces.*/
void IOWarriorDiscoverInterfaces ()
{
	USBDeviceRef			usbDeviceRef = kNoDeviceRef;
	OSErr					err = noErr;
	CFragConnectionID		usbConnID;
	CFragSymbolClass		symClass;
	THz						currentZone;
	USBDeviceDescriptorPtr		theUSBDeviceDesc;
	HIDDeviceDispatchTablePtr	theHIDDispatchTable;
	
	#ifdef IOWARRIOR_LIB_DEBUG
		printf ("IOWarriorDiscoverInterfaces:\n");
	#endif
	
	gIOWarriorCount = 0;
	
	while (noErr == err)
	{
		err = USBGetNextDeviceByClass (&usbDeviceRef, &usbConnID, kUSBHIDClass,
										kUSBAnySubClass, kUSBAnyProtocol);
		if (err)
			break;
			
		// Need to be in the system zone when we search for the symbol.
		currentZone = GetZone ();
		SetZone (SystemZone ());
		err = FindSymbol (usbConnID, "\pTheUSBDriverDescription", 
							(Ptr *)&theUSBDeviceDesc, &symClass);
		SetZone (currentZone);
		
		#ifdef IOWARRIOR_LIB_DEBUG
			printf ("USBGetNextDeviceByClass found device with vendorID %d, deviceID %d\n", theUSBDeviceDesc->vendor, theUSBDeviceDesc->product);
		#endif
		
		// If the driver that matched our device was the HID class driver,
		// it does not belong to a specific device, so it has vendor and
		// product ids of 0, 0. In that case, we have to check farther.
		if (theUSBDeviceDesc->vendor == 0 &&
			theUSBDeviceDesc->product == 0)
		{
			#ifdef IOWARRIOR_LIB_DEBUG
				printf ("Found device with vendorID 0 and productID 0, checking for TheHIDDeviceDispatchTable\n");
			#endif
		
			// Now we are going to get information from the device itself.
			currentZone = GetZone();
			SetZone(SystemZone());
			err = FindSymbol(usbConnID, "\pTheHIDDeviceDispatchTable", 
								(Ptr *)&theHIDDispatchTable, &symClass);
			SetZone(currentZone);
			if (err)
			{
				#ifdef IOWARRIOR_LIB_DEBUG
					printf ("Could not access TheHIDDeviceDispatchTable, error code %d\n", err);
					if (-2802  == err)
					{
						printf ("Reason: Symbol was not found in connection (fragSymbolNotFound).\n");
					} 
				#endif
				err = noErr;
				continue;		// There may be other devices to check.
			}

			UInt16 theVendorID = 0;
			UInt32 size = sizeof(UInt16);
			err = (*theHIDDispatchTable->pHIDGetDeviceInfo)(kHIDGetInfo_VendorID,
													&theVendorID, &size);
			if (err)
			{
				#ifdef IOWARRIOR_LIB_DEBUG
					printf ("theHIDDispatchTable->pHIDGetDeviceInfo returned error: %d\n", err );
				#endif
				err = noErr;
				continue;
			}

			UInt16 theProductID = 0;
			size = sizeof(UInt16);
			err = (*theHIDDispatchTable->pHIDGetDeviceInfo)(kHIDGetInfo_ProductID, 
													&theProductID, &size);
			if (err)
			{
				#ifdef IOWARRIOR_LIB_DEBUG
					printf ("theHIDDispatchTable->pHIDGetDeviceInfo returned error: %d\n", err );
				#endif
				err = noErr;
				continue;
			}
			#ifdef IOWARRIOR_LIB_DEBUG
				printf ("pTheHIDDeviceDispatchTable said: vendorID %d, productID %d\n",theVendorID, theProductID);
			#endif
			
			if (theVendorID == kIOWarriorVendorID && (theProductID == kIOWarrior40DeviceID) || (theProductID == kIOWarrior24DeviceID))
			{
				int interfaceType;
				
				#ifdef IOWARRIOR_LIB_DEBUG
					printf ("Adding new device to list of known IOWarriors\n");
				#endif
				// determine interface type
				if (theProductID == kIOWarrior40DeviceID)
				{
					if (gIOWarriorCount % 2 == 0)
						interfaceType = kIOWarrior40Interface0;
					else
						interfaceType = kIOWarrior40Interface1;
						
				}
				else if (theProductID == kIOWarrior24DeviceID)
				{
					if (gIOWarriorCount % 2 == 0)
						interfaceType = kIOWarrior24Interface0;
					else
						interfaceType = kIOWarrior24Interface1;
				}
				// add interface to list
				IOWarriorAddInterfaceToList (theHIDDispatchTable, interfaceType);
				gIOWarriorCount ++;
			}
		}
	}	
}

// appends a newly discovered IOWarrior interface to the linked list of known IOWarrior devices
void IOWarriorAddInterfaceToList (HIDDeviceDispatchTablePtr inInterface, int inInterfaceType)
{
    IOWarriorListNode *newNode;
    IOWarriorListNode *currentNode;
    
    // first construct a new node
    newNode =  (IOWarriorListNode*) NewPtr (sizeof (IOWarriorListNode));
    newNode->HIDDispatchTable = inInterface;
    newNode->isOpen = false;
    newNode->nextNode = NULL;
    newNode->interfaceType = inInterfaceType;

    // if the list is currently empty, the new node will become the head node
    if (NULL == gIOWarriorList)
    {
        gIOWarriorList = newNode;
    }
    else
    {
        // insert the new node at the end of the linked list
        currentNode = gIOWarriorList;
        while (NULL != currentNode->nextNode)
        {
            currentNode = currentNode->nextNode;
        }
        currentNode->nextNode = newNode;
    }
}

// clears the list of known devices
void IOWarriorClearInterfaceList ()
{
    IOWarriorListNode *currentNode;
    IOWarriorListNode *nextNode;
    
    currentNode = gIOWarriorList;
    while (currentNode)
    {
        
        nextNode = currentNode->nextNode;
        DisposePtr ((Ptr)currentNode);
        currentNode = nextNode;
    }
    gIOWarriorList = NULL;
}

// returns the hid device dispatch table at index inIndex, returns NULL if inIndex out ouf bounds
HIDDeviceDispatchTablePtr IOWarriorInterfaceAtIndex (int inIndex)
{
    IOWarriorListNode* result;
    
    result = IOWarriorListNodeAtIndex (inIndex);
    if (result)
    	return result->HIDDispatchTable;
    	
    return NULL;
}

// returns the list node element at index inIndex, returns NULL if inIndex out ouf bounds
IOWarriorListNode* IOWarriorListNodeAtIndex (int inIndex)
{
	IOWarriorListNode *currentNode;
    int			i;
    
    currentNode = gIOWarriorList;
    for (i = 0; i < inIndex && currentNode;i++)
    {
        currentNode = currentNode->nextNode;
    }
    if (currentNode)
        return currentNode;
    else
        return NULL;

}

// completion callback for HIDGetReport and HIDGetSizeReport issued by IOWarriorRead
void IOWarriorHIDReportHandlerProcPtr(void *inHIDReport, UInt32  inHIDReportLength, UInt32 /*refCon*/)
{
	//printf ("IOWarriorHIDReportHandlerProcPtr received %d bytes of data\n", inHIDReportLength);
	int byteCount;
	
	if (inHIDReportLength > gIOWarriorReadBufferSize)
		byteCount = gIOWarriorReadBufferSize;
	else
		byteCount = inHIDReportLength;
	
	BlockMove ( inHIDReport, gIOWarriorReadBuffer, gIOWarriorReadBufferSize);
	
	gIOWarriorReadOperationPending = false;
}


void IOWarriorNotificationCallback(USBDeviceNotificationParameterBlock *pb)
{			
	switch(pb->usbDeviceNotification) // why were we notified?
	{
		case kNotifyAddInterface: // because a HID device appeared
			gIOWarriorListDirty = 1;
			//DebugStr ("\pusbDeviceNotification kNotifyAddInterface");
		break;
		
		case kNotifyRemoveInterface: // because a HID device disappeared
			gIOWarriorListDirty = 1;
			//DebugStr ("\pusbDeviceNotification kNotifyRemoveInterface");
		break;
	}
}

OSErr IOWarriorInstallReportHandlerInterface1 (HIDReportHandlerProcPtr inHandlerToInstall, UInt32 inHandlerRefCon)
{
	IOWarriorListNode*			listNode;
	OSErr						err;
	
	listNode = IOWarriorFirstInterfaceNodeOfType (kIOWarrior40Interface1);

	if (listNode)
	{
		err = IOWarriorInstallReportHandlerForInterfaceAtNode (listNode, inHandlerToInstall, inHandlerRefCon);
	}
	else
		return fnfErr;
		
	return noErr;			
}

OSErr IOWarriorInstallReportHandlerForInterfaceAtNode (IOWarriorListNode *inNode, HIDReportHandlerProcPtr inHandlerToInstall, UInt32 inHandlerRefCon)
{
	OSErr						err = noErr;
	HIDDeviceDispatchTablePtr 	interface;

	interface = inNode->HIDDispatchTable;
	err = IOWarriorOpenInterfaceAtListNode (inNode);
	
	if (err)
		return err;
		
	err = (*interface->pHIDInstallReportHandler) (inNode->connectionRef, 0, inHandlerToInstall, inHandlerRefCon);
	
	return err;
}


OSErr IOWarriorRemoveReportHandlerInterface1 ()
{
	IOWarriorListNode*			listNode;
	OSErr						err = noErr;
	
	listNode = IOWarriorFirstInterfaceNodeOfType (kIOWarrior40Interface1);
	
	if (listNode)
	{
		err = IOWarriorRemoveReportHandlerForInterfaceAtNode (listNode);
	}
	else
		return fnfErr;
	
	return err;
}

OSErr IOWarriorRemoveReportHandlerForInterfaceAtNode (IOWarriorListNode *inNode)
{
	OSErr						err;
	HIDDeviceDispatchTablePtr 	interface;
		
	interface = inNode->HIDDispatchTable;

	err = (*interface->pHIDRemoveReportHandler) (inNode->connectionRef);
	
	return err;
}

void IOWarriorSetAutoCloseInterfaces (int inFlag)
{
	gIOWarriorAutoCloseInterfaces = inFlag;
}

void IOWarriorCloseAllInterfaces ()
{
	int i, limit;
	
	limit = IOWarriorCount ();
	for (i = 0; i < limit; i++)
	{
		IOWarriorListNode* node;
		
		node = IOWarriorListNodeAtIndex (i);
		if (node)
		{
			IOWarriorCloseInterfaceAtListNode (node);
		}
	}
}